<?php

class Meks_Instagram_Widget extends WP_Widget {

	/**
	 * Unique identifier for your widget.
	 *
	 * @since    1.0.0
	 * @var      string
	 */
	protected $widget_slug = 'meks_instagram';

	/**
	 * Unique identifier for localization.
	 *
	 * @since    1.0.0
	 * @var      string
	 */
	protected $widget_text_domain = 'meks-easy-instagram-widget';

	/**
	 * Default value holder
	 *
	 * @since    1.0.0
	 * @var      array
	 */
	protected $defaults;

	/**
	 * Access Token
	 *
	 * @since    1.0.0
	 * @var      array
	 */
	private $access_token;

	/**
	 * Username
	 *
	 * @since    1.0.0
	 * @var      array
	 */
	private $username;

	/**
	 * Specifies the class name and description, instantiates the widget and includes necessary stylesheets and JavaScript.
	 */
	public function __construct() {

		parent::__construct(
			$this->widget_slug,
			__( 'Meks Easy Instagram Widget', 'meks-easy-instagram-widget' ),
			array(
				'description' => __( 'Easily display Instagram photos with this widget.', 'meks-easy-instagram-widget' ),
			)
		);

		$this->defaults = array(
			'title'            => 'Instagram',
			'username_hashtag' => '',
			'photos_number'    => 9,
			'columns'          => 3,
			'photo_space'      => 1,
			'container_size'   => 300,
			'transient_time'   => DAY_IN_SECONDS,
			'link_text'        => __( 'Follow', 'meks-easy-instagram-widget' ),
		);

		// Allow themes or plugins to modify default parameters
		$this->defaults = apply_filters( 'meks_instagram_widget_modify_defaults', $this->defaults );

		// Register site styles and scripts
		add_action( 'wp_enqueue_scripts', array( $this, 'register_widget_styles' ) );

		// Register admin styles and scripts
		add_action( 'admin_enqueue_scripts', array( $this, 'register_admin_script' ) );

	}

	public function is_authorized(){

		return !empty($this->access_token);
	}


	/**
	 * Outputs the content of the widget.
	 *
	 * @param array   args  The array of form elements
	 * @param array   instance The current instance of the widget
	 */
	public function widget( $args, $instance ) {

		$widget_settings    = get_option( 'meks_instagram_settings' );
		$this->access_token = isset($widget_settings['access_token']) ? $widget_settings['access_token'] : '';

		extract( $args, EXTR_SKIP );
		$instance = wp_parse_args( (array) $instance, $this->defaults );

		$title = apply_filters( 'widget_title', $instance['title'] );

		echo $before_widget;

		if ( ! empty( $title ) ) {
			echo $before_title . $title . $after_title;
		}

		$photos = $this->get_photos( $instance['username_hashtag'], $instance['transient_time'] );

		if ( is_wp_error( $photos ) ) {
			echo $photos->get_error_message();
			echo $after_widget;
			return;
		}

		$photos = $this->limit_images_number( $photos, $instance['photos_number'] );
		$size   = $this->calculate_image_size( $instance['container_size'], $instance['photo_space'], $instance['columns'] );

		$follow_link = $this->get_follow_link( $instance['username_hashtag'] );

		ob_start();
		include $this->get_template( MEKS_INSTAGRAM_WIDGET_DIR . 'views/widget_html' );
		$widget_content = ob_get_clean();

		echo $widget_content;
		echo $after_widget;

	}


	/**
	 * Processes the widget options to be saved.
	 *
	 * @param array   new_instance The new instance of values to be generated via the update.
	 * @param array   old_instance The previous instance of values before the update.
	 */
	public function update( $new_instance, $old_instance ) {

		$instance                     = array();
		$instance['title']            = strip_tags( $new_instance['title'] );
		$instance['username_hashtag'] = strip_tags( $new_instance['username_hashtag'] );
		$instance['photos_number']    = absint( $new_instance['photos_number'] );
		$instance['columns']          = absint( $new_instance['columns'] );
		$instance['photo_space']      = absint( $new_instance['photo_space'] );
		$instance['container_size']   = absint( $new_instance['container_size'] );
		$instance['transient_time']   = absint( $new_instance['transient_time'] );
		$instance['link_text']        = strip_tags( $new_instance['link_text'] );

		return $instance;

	}

	/**
	 * Generates the administration form for the widget.
	 *
	 * @param array   instance The array of keys and values for the widget.
	 */
	public function form( $instance ) {

		$instance = wp_parse_args( (array) $instance, $this->defaults );

		include MEKS_INSTAGRAM_WIDGET_DIR . 'views/admin_html.php';

	}

	/**
	 * Get predefined template which can be overridden by child themes
	 *
	 * @since 1.0.0
	 *
	 * @param string $template
	 * @return string      - File Path
	 */
	private function get_template( $template ) {
		$template_slug = rtrim( $template, '.php' );
		$template      = $template_slug . '.php';

		if ( $theme_file = locate_template( array( '/core/widgets/' . $template ) ) ) :
			$file = $theme_file;
		else :
			$file = $template;
		endif;

		return $file;
	}

	/**
	 * Get photos form Instagram base on username or hashtag with transient caching for one day
	 *
	 * @since 1.0.0
	 *
	 * @param string $username_or_hashtag Searched username or hashtag
	 * @param int    $transient_time Time in seconds
	 * @return array  List of all photos sizes with additional information
	 */
	protected function get_photos( $usernames_or_hashtags, $transient_time ) {

		if ( empty( $usernames_or_hashtags ) ) {
			return false;
		}

		$transient_key = $this->generate_transient_key( $usernames_or_hashtags );

		$cached = get_transient( $transient_key );

		if ( ! empty( $cached ) ) {
			return $cached;
		}

		$usernames_or_hashtags = explode( ',', $usernames_or_hashtags );

		if ( $this->is_authorized() && count( $usernames_or_hashtags ) > 1 ) {
			$usernames_or_hashtags = str_replace( '@', '', current( $usernames_or_hashtags ) );
			$usernames_or_hashtags = array( $usernames_or_hashtags );

		}

		$images = array();

		foreach ( $usernames_or_hashtags as  $username_or_hashtag ) {

			$this->username = trim( $username_or_hashtag );

			$data = $this->get_instagram_data();

			if ( is_wp_error( $data ) ) {
				return $data;
			}

			$images[] = $data;
		}

		$images = array_reduce( $images, 'array_merge', array() );

		usort(
			$images,
			function ( $a, $b ) {
				if ( $a['time'] == $b['time'] ) {
					return 0;
				}
				return ( $a['time'] < $b['time'] ) ? 1 : -1;
			}
		);

		if ( !$this->is_authorized() && $transient_time < ( 12 * HOUR_IN_SECONDS ) ) {
			$transient_time = DAY_IN_SECONDS;
		}

		set_transient( $transient_key, $images, $transient_time );

		return $images;
	}


	/**
	 * Generates transient key to cache the results
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	function generate_transient_key( $usernames_or_hashtags ) {

		$transient_key = md5( 'meks_instagram_widget_' . $usernames_or_hashtags );

		return $transient_key;

	}


	/**
	 * Function to return endpoint URL or simple URL for follow link
	 *
	 * @since 1.0.0
	 *
	 * @param string $searched_term
	 * @param string $proxy
	 * @return string    - URL
	 */
	protected function get_instagram_url() {

		$searched_term = trim( strtolower( $this->username ) );

		switch ( substr( $searched_term, 0, 1 ) ) {
			case '#':
				$url = 'https://instagram.com/explore/tags/' . str_replace( '#', '', $searched_term );
				break;

			default:
				$url = 'https://instagram.com/' . str_replace( '@', '', $searched_term );
				break;
		}

		return $url;
	}


	/**
	 * Make remote request base on Instagram endpoints, get JSON and collect all images.
	 *
	 * @since 1.0.0
	 *
	 * @return array  - List of collected images
	 */
	protected function get_instagram_data() {

		if ( !$this->is_authorized() ) {
			return $this->get_instagram_data_without_token();
		}

		return $this->get_instagram_data_with_token();

	}

	/**
	 * Make request with token.
	 *
	 * @since 1.0.0
	 *
	 * @return array List of collected images
	 */
	protected function get_instagram_data_with_token() {

		$response = wp_remote_get( sprintf( 'https://api.instagram.com/v1/users/self/media/recent/?access_token=%s', $this->access_token ) );

		if ( is_wp_error( $response ) || 200 != wp_remote_retrieve_response_code( $response ) ) {
			return new WP_Error( 'invalid-token', esc_html__( 'Invalid username or token.', 'meks-easy-instagram-widget' ) );
		}

		$data           = json_decode( wp_remote_retrieve_body( $response ) );
		$token_username = ! empty( $data->data[0]->user->username ) ? $data->data[0]->user->username : '';

		$this->username = str_replace( '@', '', $this->username );

		if ( ! empty( $token_username ) && ! empty( $this->username ) ) {

			if ( $this->username !== $token_username ) {

				$data = $this->get_instagram_data_without_token();

				if ( ! empty( $data ) ) {
					return $data;
				}

				if ( empty( $data ) ) {
					return new WP_Error( 'authorized-user', esc_html__( 'Invalid username. Authorized users can only pull images from their own account.', 'meks-easy-instagram-widget' ) );
				}
			}
		}

		$images = $this->parse_instagram_images_with_token( $data );

		if ( empty( $images ) ) {
			return new WP_Error( 'no_images', esc_html__( 'Images not found. This may be a temporary problem. Please try again soon.', 'meks-easy-instagram-widget' ) );
		}

		return $images;

	}


	/**
	 * Make request without token
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	protected function get_instagram_data_without_token() {

		$url = $this->get_instagram_url();

		
		

		$request = wp_remote_get( $url );
		$body    = wp_remote_retrieve_body( $request );

		$shared = explode( 'window._sharedData = ', $body );
		$json   = explode( ';</script>', $shared[1] );
		$data   = json_decode( $json[0], true );

		if ( empty( $data ) ) {
			return new WP_Error( 'blocked',  sprintf( esc_html__('Instagram has returned empty data. Please authorize your Instagram account in the %s plugin settings %s.', 'meks-easy-instagram-widget' ), '<a href="'.esc_url( admin_url( 'options-general.php?page=meks-instagram' ) ).'">', '</a>') );
		}

		if ( isset( $data['entry_data']['ProfilePage'][0]['graphql']['user']['edge_owner_to_timeline_media']['edges'] ) ) {
			$images = $data['entry_data']['ProfilePage'][0]['graphql']['user']['edge_owner_to_timeline_media']['edges'];
		} elseif ( isset( $data['entry_data']['TagPage'][0]['graphql']['hashtag']['edge_hashtag_to_media']['edges'] ) ) {
			$images = $data['entry_data']['TagPage'][0]['graphql']['hashtag']['edge_hashtag_to_media']['edges'];
		} else {
			return new WP_Error( 'blocked',  sprintf( esc_html__('Instagram has returned empty data. Please authorize your Instagram account in the %s plugin settings %s.', 'meks-easy-instagram-widget' ), '<a href="'.esc_url( admin_url( 'options-general.php?page=meks-instagram' ) ).'">', '</a>') );
		}

		
		$images = $this->parse_instagram_images_without_token( $images );
		
		if ( empty( $images ) ) {
			return new WP_Error( 'no_images', esc_html__( 'Images not found. This may be a temporary problem. Please try again soon.', 'meks-easy-instagram-widget' ) );
		}

		return $images;

	}


	/**
	 * Parse instagram images
	 *
	 * @since  1.0.1
	 *
	 * @param array $data
	 * @return array  - List of images prepared for displaying
	 */
	protected function parse_instagram_images_with_token( $data ) {

		$pretty_images = array();

		foreach ( $data->data as $image ) {

			$pretty_images[] = array(
				'caption'   => isset( $image->caption->text ) ? $image->caption->text : '',
				'link'      => trailingslashit( $image->link ),
				'time'      => $image->created_time,
				'comments'  => $image->comments->count,
				'likes'     => $image->likes->count,
				'thumbnail' => $image->images->thumbnail->url, // 150x150
				'small'     => $image->images->low_resolution->url, // 320x320
				'medium'    => $image->images->low_resolution->url, // 320x320
				'large'     => $image->images->standard_resolution->url, // 640x640
				'original'  => $image->images->standard_resolution->url, // 640x640
			);

		}

		return $pretty_images;
	}


	/**
	 * Parse instagram images
	 *
	 * @since  1.0.1
	 *
	 * @param array $images - Raw Images
	 * @return array           - List of images prepared for displaying
	 */
	protected function parse_instagram_images_without_token( $images ) {

		$pretty_images = array();

		foreach ( $images as $image ) {

			$pretty_images[] = array(
				'caption'   => isset( $image['node']['edge_media_to_caption']['edges'][0]['node']['text'] ) ? $image['node']['edge_media_to_caption']['edges'][0]['node']['text'] : '',
				'link'      => trailingslashit( 'https://instagram.com/p/' . $image['node']['shortcode'] ),
				'time'      => $image['node']['taken_at_timestamp'],
				'comments'  => $image['node']['edge_media_to_comment']['count'],
				'likes'     => $image['node']['edge_liked_by']['count'],
				'thumbnail' => preg_replace( '/^https?\:/i', '', $image['node']['thumbnail_resources'][0]['src'] ), // 150
				'small'     => preg_replace( '/^https?\:/i', '', $image['node']['thumbnail_resources'][1]['src'] ), // 240
				'medium'    => preg_replace( '/^https?\:/i', '', $image['node']['thumbnail_resources'][2]['src'] ), // 320
				'large'     => preg_replace( '/^https?\:/i', '', $image['node']['thumbnail_resources'][3]['src'] ), // 480
				'original'  => preg_replace( '/^https?\:/i', '', $image['node']['display_url'] ),
			);

		}

		return $pretty_images;
	}


	/**
	 * Limit number of displayed images on front-end
	 *
	 * @since  1.0.0
	 *
	 * @param array  $photos - Lists of images
	 * @param number $limit  - Max number of image that we want to show
	 * @return array             - Limited List
	 */
	protected function limit_images_number( $photos, $limit = 1 ) {
		if ( empty( $photos ) || is_wp_error( $photos ) ) {
			return array();
		}
		return array_slice( $photos, 0, $limit );
	}

	/**
	 * Calculate which images size  to use base on container width and photo space and calculate images columns
	 *
	 * @since  1.0.0
	 *
	 * @param int $container_size
	 * @param int $photo_space
	 * @param int $columns
	 * @return array              - Proper image size and flex column calculation
	 */
	public function calculate_image_size( $container_size, $photo_space, $columns ) {

		$width = ( $container_size - ( $photo_space * ( $columns - 1 ) ) ) / $columns;
		$flex  = 100 / $columns;

		$size         = array();
		$size['flex'] = $flex;

		switch ( $width ) {

			case $width <= 150:
				$size['thumbnail'] = 'thumbnail';
				break;

			case $width <= 240:
				$size['thumbnail'] = 'small';
				break;

			case $width <= 320:
				$size['thumbnail'] = 'medium';
				break;

			case $width <= 480:
				$size['thumbnail'] = 'large';
				break;

			default:
				$size['thumbnail'] = 'original';
				break;
		}

		return $size;

	}

	/**
	 * Check if is one username or hashtag.
	 *
	 * @since 1.0.0
	 *
	 * @param string $usernames_or_hashtags String from username or hashtag input field
	 * @return string    Follow URL or empty string
	 */
	protected function get_follow_link( $usernames_or_hashtags ) {

		$usernames_hashtags_array   = explode( ',', $usernames_or_hashtags );
		$usernames_hashtags   = str_replace( '@', '', current( $usernames_hashtags_array ) );
		$this->username = $usernames_hashtags;
		
		return $this->get_instagram_url();
	}


	/**
	 * Registers and enqueue widget-specific styles.
	 */
	public function register_widget_styles() {
		wp_enqueue_style( $this->widget_slug . '-widget-styles', MEKS_INSTAGRAM_WIDGET_URL . 'css/widget.css' );
	}

	/**
	 * Registers and enqueue admin specific scripts.
	 */
	public function register_admin_script() {
		wp_enqueue_script( $this->widget_slug . '-admin-script', MEKS_INSTAGRAM_WIDGET_URL . 'js/admin.js', true, MEKS_INSTAGRAM_WIDGET_VER );
		wp_enqueue_style( $this->widget_slug . '-admin-styles', MEKS_INSTAGRAM_WIDGET_URL . 'css/admin.css' );
	}


}
